<?php

/*
 * The Black Sheep Research Plugin for Restful in Joomla
 *
 * This plugin enables the handling of restful style requests in Joomla. One or
 * more additional plugins must be deployed to implement specific services. The
 * service plugins must be in the folder restful, and provide details of the
 * URIs and calls to be handled in response to the onRestfulPaths trigger. An
 * example is:
 *
  public function onRestfulPaths($context, &$paths) {
  if ('plg_system_restful' == $context) {
  $paths['/myrestful/myaction'] = array('POST' => array($this, 'myaction'));
  $paths['/myrestful/test'] = array('GET' => array($this, 'test'));
  }
  }
 *
 * The way this works is that if, for example, a POST request is made to
 * https://example.com/myrestful/myaction then the callable array($this, 'myaction')
 * is invoked. This means that the service plugin must have a method
 * called myaction. The call does not have to be to the service plugin, it can be
 * to anywhere that the service plugin can construct a callable for.
 *
 * So the service plugin might implement a method:
 *
  public function myaction($uri, $restful) {
  echo "The URI $uri resulted in myaction being called";
  // Do something
  }

 * The $uri in this case willl contain "/myrestful/myaction". The instance of
 * this plugin is passed as the second parameter, $restful. It can be used to call
 * the helpful methods shown below and listed here as:
 *
 * authenticateUser($username, $password)
 * authenticateUserAndGroup($username, $password, $groups)
 * getHeaderUserPass()
 * respondAndExit($response_object, $response_code, $response_message)
 *
 * The first two of these return true or false. The third will return an array
 * with two elements - username and password - provided a valid Basic
 * Authorization header is received. The fourth accepts an arbitrary object
 * that is converted to JSON and returned as the HTTP response, along with a
 * result header based on the provided response code and message.
 *
 * Note that for getHeaderUserPass to work, an HTTP Basic Authorization header
 * must be supplied by the caller. Basic Authorization must be used only with
 * SSL as otherwise username and password are being passed insecurely.
 *
 * Copyright (c) 2022 Black Sheep Research
 * Author: Martin Brampton martin@black-sheep-research.com
 * Web: https://black-sheep-research.com
 * License: GNU General Public License version 2
 */

use Joomla\CMS\Plugin\CMSPlugin;

\defined('_JEXEC') or die;

class PlgSystemUpgrader extends CMSPlugin {

    public function onInstallerBeforePackageDownload($context, &$args) {
        $url = $args[0];
        foreach ($this->params->get('servers', []) as $server) {
            $serverUrl = rtrim($server->url, '/');
            $hashfield = $server->hashfield ?? 'hash';
            $hash = $server->hash ?? '';

            if (!empty($hash) && strpos($url, $serverUrl) === 0) {
                $separator = (strpos($url, '?') === false) ? '?' : '&';
                $args[0] .= $separator . $hashfield . '=' . urlencode($hash);
                break; // Stop after first match
            }
        }
    }
}
