<?php

/**
 * Kunena Search Plugin
 *
 * @package       Kunena.plg_search_kunena
 *
 * @copyright (C) 2008 - 2025 Kunena Team, Martin Brampton. All rights reserved.
 * @license       http://www.gnu.org/copyleft/gpl.html GNU/GPL
 * @link          https://www.remository.com
 * */
defined('_JEXEC') or die();

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Kunena\Forum\Libraries\Factory\KunenaFactory;
use Kunena\Forum\Libraries\Forum\KunenaForum;
use Kunena\Forum\Libraries\Forum\Message\KunenaMessageHelper;
use Kunena\Forum\Libraries\Html\KunenaParser;

JLoader::register('RemsearchHelper', JPATH_ADMINISTRATOR . '/components/com_remsearch/helpers/remsearch.php');

class PlgSearchKunena extends CMSPlugin {

    private static $checked = false;
    private static $present = false;

    private function isKunenaPresent(): bool {
        if (self::$checked) {
            return self::$present;
        } else {
            self::$checked = true;
        }

        // Kunena detection and version check
        $minKunenaVersion = '6.0';

        if (!class_exists(KunenaForum::class) || !KunenaForum::isCompatible($minKunenaVersion)) {
            return self::$present;
        }

        // Kunena online check
        if (!KunenaForum::enabled()) {
            return self::$present;
        }

        // Setup Kunena framework
        self::$present = true;
        KunenaForum::setup();
        KunenaFactory::loadLanguage('com_kunena.sys', 'admin');
        return self::$present;
    }

    /**
     * Determine areas searchable by this plugin.
     *
     * @return  array  An array of search areas.
     *
     * @since   1.6
     */
    public function onContentSearchAreas() {
        static $areas = array();

        if ($this->isKunenaPresent() && empty($areas)) {
            $areas['kunena'] = Text::_('COM_KUNENA');
        }

        return $areas;
    }

    /**
     * @param        $text
     * @param   string $phrase
     * @param   string $ordering
     * @param   null   $areas
     *
     * @return array
     */
    public function onContentSearch($text, $phrase = '', $ordering = '', $areas = null) {
        if (!$this->isKunenaPresent()) {
            return [];
        }

        // If the array is not correct, return it:
        if (is_array($areas) && !array_intersect($areas, array_keys($this->onContentSearchAreas()))) {
            return array();
        }

        // And define the parameters. For example like this..
        $limit = $this->params->get('search_limit', 50);
        $contentLimit = $this->params->get('content_limit', 40);
        $bbcode = $this->params->get('show_bbcode', 1);
        $openInNewPage = $this->params->get('open_new_page', 1);

        // Use the function trim to delete spaces in front of or at the back of the searching terms
        $text = trim($text);

        // Return Array when nothing was filled in
        if ($text == '') {
            return array();
        }

        $db = Factory::getDbo();

        // After this, you have to add the database part. This will be the most difficult part, because this changes per situation.
        // In the coding examples later on you will find some of the examples used by Joomla! 1.5 core Search Plugins.
        // It will look something like this.
        switch ($phrase) {
            // Search exact
            case 'exact' :
                $text = $db->quote('%' . $db->escape($text) . '%', false);
                $where = "(m.subject LIKE {$text} OR t.message LIKE {$text})";
                break;

            // Search all or any
            case 'all' :
            case 'any' :
            default :
                $where = array();
                $words = explode(' ', $text);

                foreach ($words as $word) {
                    $word = $db->quote('%' . $db->escape(trim($word)) . '%', false);
                    $where [] = "m.subject LIKE {$word} OR t.message LIKE {$word}";
                }

                $where = '(' . implode(($phrase == 'all' ? ') AND (' : ') OR ('), $where) . ')';
                break;
        }

        // Ordering of the results
        switch ($ordering) {
            // Oldest first
            case 'oldest' :
                $orderby = 'm.time ASC';
                break;

            // Popular first
            case 'popular' :
                $orderby = 'm.hits DESC, m.time DESC';
                break;

            // Newest first
            case 'newest' :
                $orderby = 'm.time DESC';
                break;

            // Alphabetic, ascending
            case 'alpha' :
            // Default setting: alphabetic, ascending
            default :
                $orderby = 'm.subject ASC, m.time DESC';
        }

        $params = array('orderby' => $orderby, 'where' => $where, 'starttime' => -1);
        list($total, $messages) = KunenaMessageHelper::getLatestMessages(false, 0, $limit, $params);
        $rows = [];

        foreach ($messages as $message) {
            /** @var KunenaForumMessage $message */
            // Function must return: href, title, section, created, text, browsernav
            $row = new StdClass;
            $row->id = $message->id;
            $row->href = $message->getUrl();
            $row->title = substr($message->subject, '0', $contentLimit);
            $row->section = $message->getCategory()->name;
            $row->created = $message->time;

            if ($bbcode) {
                $row->text = KunenaParser::parseBBCode($message->message, $contentLimit);
            } else {
                $row->text = KunenaParser::stripBBCode($message->message, $contentLimit);
            }

            $row->browsernav = $openInNewPage ? 1 : 0;
            $rows[] = $row;
        }

        // Return the search results in an array
        return $rows;
    }
}
