<?php

namespace BlackSheepResearch\Plugin\System\Diagnostic;

use Joomla\CMS\Factory;
use Joomla\CMS\Object\CMSObject;

\defined('_JEXEC') or die;

class Recorder {

    private static $diagnostics = [];
    private static $ipaddress = '';
    private static $active = null;
    private static $trace = null;

    private static function priv_active(): bool {
        if (!isset(self::$active)) {
            $params = \class_exists(CMSObject::class) ? new CMSObject : new \JObject();
            $args = [];
            $app = Factory::getApplication();
            self::$active = $app->triggerEvent('onIsActive', ['plg_diagnostic', $args, $params])[0];
            self::$trace = $app->triggerEvent('onTraceAll', ['plg_diagnostic', $args, $params])[0];
        }
        return self::$active;
    }

    public static function active() : bool {
        return self::priv_active();
    }

    public static function store(string $info, bool $trace = false): void {
        if (self::active()) {
            $will_trace = $trace || self::$trace;
            self::$diagnostics[] = $info . self::tracker($will_trace);
            if ($will_trace) {
                self::trace();
            }
        }
    }

    private static function tracker(bool $trace): string {
        if (!$trace) {
            $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 2);
            $line = $backtrace[1]['line'] ?? 'unknown';
            $file = $backtrace[1]['file'] ?? 'unknown';
            $message = '<br /><span style="background-color: #f2f2f2; padding: 4px;"><em> (Line %d in %s)</em></span>';
            return sprintf($message, $line, $file);
        }
    }

    public static function trace(): void {
        if (self::active()) {
            static $counter = 0;
            $html = ['TRACE:'];
            foreach (debug_backtrace() as $back) {
                if (isset($back['file']) AND $back['file']) {
                    $html[] = $back['file'] . ':' . $back['line'];
                }
            }
            $counter++;
            self::$diagnostics[] = implode("\n<br />", ($html ?? []));
            if (1000 < $counter) {
                die('Trace killed - Probably looping');
            }
        }
    }

    public static function dump(...$args): void {
        if (self::active()) {
            ob_start();
            foreach ($args as $variable) {
                var_dump($variable);
            }
            self::$diagnostics[] = ob_get_clean() . self::tracker(false);
        }
    }

    public static function die(string $reason = ''): void {
        if (self::active()) {
            echo implode("<br />\n", self::$diagnostics);
            die("<br />\n" . ($reason ?: 'Killed on request to diagnostic recorder') . self::tracker(false));
        }
    }

    public static function get(): string {
        return self::active() ? implode("<br />\n", self::$diagnostics) : '';
    }

    public static function getFormatted() : string {
        $data = self::get();
        return <<<DIAGNOSE

<div style="border: 2px solid black; padding: 10px;">
  <h4 style="background-color: #f2f2f2; padding: 10px;">
    Information from diagnostics passed to the Recorder class
  </h4>
  <div>
    $data
  </div>
  <br /><hr>
  <div>
    <strong>use BlackSheepResearch\Plugin\System\Diagnostic\Recorder;</strong>
    <br />public static function active() : bool
    <br />public static function store(string \$info, bool \$trace = false): void
    <br />public static function trace(): void
    <br />public static function dump(...\$args): void
    <br />public static function die(string \$reason = ''): void
  </div>
</div>

DIAGNOSE;

    }

    public static function displayError (object $error) : string {
        $trace = nl2br($error->getTraceAsString());
        $previous = $error->getPrevious() ? $error->getPrevious()->getMessage() : 'None';
        return <<<ERRORCODE

<div style="border: 2px solid black; padding: 10px; margin-bottom:20px;">
  <h2 style="background-color: #f2f2f2; padding: 10px;">
    Error: {$error->getMessage()}
  </h2>
  <div>
    Exception Code: {$error->getCode()}<br />
    Exception File: {$error->getFile()}<br />
    Exception Line: {$error->getLine()}<br />
    Exception Trace: <br />{$trace}<br />
    Exception Previous: {$previous}<br />
  </div>
</div>

ERRORCODE;

    }

}
