<?php

/*
 * The Black Sheep Research Plugin for Restful in Joomla
 *
 * This plugin enables the handling of restful style requests in Joomla. One or
 * more additional plugins must be deployed to implement specific services. The
 * service plugins must be in the folder restful, and provide details of the
 * URIs and calls to be handled in response to the onRestfulPaths trigger. An
 * example is:
 *
  public function onRestfulPaths($context, &$paths) {
  if ('plg_system_restful' == $context) {
  $paths['/myrestful/myaction'] = array('POST' => array($this, 'myaction'));
  $paths['/myrestful/test'] = array('GET' => array($this, 'test'));
  }
  }
 *
 * The way this works is that if, for example, a POST request is made to
 * https://example.com/myrestful/myaction then the callable array($this, 'myaction')
 * is invoked. This means that the service plugin must have a method
 * called myaction. The call does not have to be to the service plugin, it can be
 * to anywhere that the service plugin can construct a callable for.
 *
 * So the service plugin might implement a method:
 *
  public function myaction($uri, $restful) {
  echo "The URI $uri resulted in myaction being called";
  // Do something
  }

 * The $uri in this case willl contain "/myrestful/myaction". The instance of
 * this plugin is passed as the second parameter, $restful. It can be used to call
 * the helpful methods shown below and listed here as:
 *
 * authenticateUser($username, $password)
 * authenticateUserAndGroup($username, $password, $groups)
 * getHeaderUserPass()
 * respondAndExit($response_object, $response_code, $response_message)
 *
 * The first two of these return true or false. The third will return an array
 * with two elements - username and password - provided a valid Basic
 * Authorization header is received. The fourth accepts an arbitrary object
 * that is converted to JSON and returned as the HTTP response, along with a
 * result header based on the provided response code and message.
 *
 * Note that for getHeaderUserPass to work, an HTTP Basic Authorization header
 * must be supplied by the caller. Basic Authorization must be used only with
 * SSL as otherwise username and password are being passed insecurely.
 *
 * Copyright (c) 2022-5 Black Sheep Research
 * Author: Martin Brampton martin@black-sheep-research.com
 * Web: https://black-sheep-research.com
 * License: GNU General Public License version 2
 */

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Object\CMSObject;
use Joomla\CMS\User\UserHelper;
use Joomla\CMS\Helper\UserGroupsHelper;
use Joomla\CMS\User\UserFactory;

\defined('_JEXEC') or die;

class PlgSystemRestful extends CMSPlugin {

    private $paths = array();
    private $user = null;
    private $method = '';
    private $querystring = '';
    public $params = null;
    private $input = null;

    public function __construct(&$subject, $config) {
        $this->params = new CMSObject;
        PluginHelper::importPlugin('restful');
        parent::__construct($subject, $config);
        $this->input = Factory::getApplication()->getInput();
    }

    public function onAfterInitialise() {
        $this->method = $this->input->server->get('REQUEST_METHOD', '', 'string');
        $requestUri = $this->input->server->get('REQUEST_URI', '', 'string');
        $parts = explode('?', $requestUri);
        $uri = $parts[0];
        $this->querystring = $parts[1] ?? '';

        foreach ($this->findRestfulPlugins($uri) as $action) {
            if (isset($action[$this->method])) {
                call_user_func($action[$this->method], $uri, $this);
            }
        }
    }

    private function findRestfulPlugins($uri) {
        // Use the old-style triggerEvent for backwards compatibility
        // This passes arguments by reference, which old plugins expect
        Factory::getApplication()->triggerEvent('onRestfulPaths', [
            'plg_system_restful',
            &$this->paths,
            &$this->params,
            0
        ]);

        $matches = array_filter(array_keys($this->paths), function ($var) use ($uri) {
            return 0 === stripos($uri, $var);
        });
        return array_intersect_key($this->paths, array_flip($matches));
    }

    public function authenticateUser($username, $password, $super = false) {
        // Get UserFactory from the DI container
        $userFactory = Factory::getContainer()->get(UserFactory::class);
        // Load user by username
        $this->user = $userFactory->loadUserByUsername($username);
        // Check if user exists (loadUserByUsername returns a guest user if not found)
        if ($this->user->guest) {
            return false;
        }
        // Verify password
        if (!UserHelper::verifyPassword($password, $this->user->password, $this->user->id)) {
            return false;
        }
        // Check super user authorization if required
        return !$super || $this->user->authorise('core.admin');
    }

    public function authenticateUserAndGroup($username, $password, $groups) {
        if ($this->authenticateUser($username, $password)) {
            foreach (UserGroupsHelper::getInstance()->getAll() as $group) {
                if (in_array($group->title, (array) $groups) AND isset($this->user->groups[$group->id])) {
                    return true;
                }
            }
        }
        return false;
    }

    /*     * *************************************************************************
     *
     * PLEASE NOTE: base64 encoding is a necessary part of
     * Basic HTTP Authorization.
     *
     * ************************************************************************ */

    public function getHeaderUserPass() {
        // Joomla's Input handles HTTP_AUTHORIZATION and REDIRECT_HTTP_AUTHORIZATION automatically
        $authHeader = $this->input->server->get('HTTP_AUTHORIZATION', '', 'string');
        // Fallback to native PHP functions if needed
        if (!$authHeader && function_exists('getallheaders')) {
            $headers = getallheaders();
            $authHeader = $headers['Authorization'] ?? '';
        }
        // Parse Basic Authorization
        if ($authHeader && preg_match('/^Basic\s+(.+)$/i', $authHeader, $matches)) {
            return explode(':', base64_decode($matches[1]), 2);
        }
        return ['', ''];
    }

    public function respondAndExit($response_object, $response_code, $response_message) {
        header($this->input->server->get('SERVER_PROTOCOL', 'HTTP/1.1', 'string') . " $response_code $response_message");
        header('Content-Type: application/json');
        echo json_encode($response_object);
        exit;
    }

    public function getQueries() {
        $queries = [];
        parse_str($this->querystring, $queries);
        return $queries;
    }
}
